package com.actyve.openapi.sdk.client;

import com.actyve.openapi.sdk.enums.ResponseCodeEnum;
import com.actyve.openapi.sdk.utils.JsonUtils;
import com.actyve.openapi.sdk.utils.OkHttpUtil;
import com.actyve.openapi.sdk.utils.SignUtils;
import com.actyve.openapi.sdk.utils.StringUtils;
import com.alibaba.fastjson.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

public class ActyveClient {

    private static final Logger log = LoggerFactory.getLogger(ActyveClient.class);

    private final ClientConfig clientConfig;

    public ActyveClient(ClientConfig clientConfig) {
        this.clientConfig = clientConfig;
    }

    public <T extends ActyveResponse> T execute(ActyveRequest<T> request) {
        Endpoint endPoint = request.getEndPointEnum();
        Map<String, String> h = request.getHeader();
        request.setHeader(null);
        String requestBody = JSONObject.toJSONString(request);
        String url = clientConfig.getDomainUrl() + endPoint.getUrl();
        String httpResponse = null;
        Exception exp = null;
        Long startL = System.currentTimeMillis();
        try {
            Map<String, String> headers = header(h, endPoint, requestBody);

            if (clientConfig.isDebug()) {
                log.info("Start request Actyve API:{},header:{},body:{}", request.getEndPointEnum().getUrl(), JsonUtils.toJson(headers), requestBody);
            }

            if (request.getEndPointEnum().getMethod().equals(Constants.HttpMethod.POST)) {
                httpResponse = OkHttpUtil.postJson(url, headers, requestBody);
            } else if (request.getEndPointEnum().getMethod().equals(Constants.HttpMethod.GET)) {
                Map<String, Object> paramMap = JsonUtils.toMap(requestBody);
                httpResponse =
                        OkHttpUtil.get(url, headers, paramMap.entrySet().stream().filter(e -> e.getValue() != null).collect(Collectors.toMap(Map.Entry::getKey, e -> String.valueOf(e.getValue()))));
            }
        } catch (Exception e) {
            exp = e;
        }
        log.info(httpResponse);
        T t = (T) readResponse(httpResponse, getResponseClass(request), exp);
        if (clientConfig.isDebug()) {
            log.info("{} request Actyve API:{},cost:{} ms,httpResponse:{}", t.isSuccess() ? "Success" : "Failure", request.getEndPointEnum().getUrl(), System.currentTimeMillis() - startL,
                    httpResponse);
        }
        return t;
    }

    private <T extends ActyveResponse> T readResponse(String httpResponse, Class<T> clazz, Exception e) {
        if (e != null) {
            log.error(e.getMessage(), e);
            if (e.getMessage().contains("Read timed out")) {
                return buildErrorResponse(ResponseCodeEnum.S_Common_NetworkError.getCode(), e.getMessage(), "", clazz);
            } else {
                return buildErrorResponse(ResponseCodeEnum.S_Common_SystemError.getCode(), e.getMessage(), "", clazz);
            }
        }

        if (StringUtils.isEmpty(httpResponse)) {
            return buildErrorResponse(ResponseCodeEnum.S_Common_NetworkError.getCode(), ResponseCodeEnum.S_Common_NetworkError.getMsg(), "", clazz);
        }

        Map rspMap = JsonUtils.parse(httpResponse, Map.class);
        try {
            if (Constants.Response.success.equals(rspMap.get(Constants.Response.code).toString())) {
                Object responseData = rspMap.get(Constants.Response.data);
                if (responseData instanceof Map) {
                    Map<String, Object> map = (Map<String, Object>) responseData;

                    // 获取原始的 pageSize 和 pageNo
                    Object pageSizeObj = map.get("pageSize");
                    Object pageNoObj = map.get("pageNo");

                    // 检查 pageSize 和 pageNo 是否存在，并且类型为整数
                    if (pageSizeObj instanceof Integer && pageNoObj instanceof Integer) {
                        int pageSize = (Integer) pageSizeObj;
                        int pageNo = (Integer) pageNoObj;

                        // 替换属性名为 page_size 和 page_no
                        map.put("page_size", pageSize);
                        map.put("page_no", pageNo);

                        // 移除原始的 pageSize 和 pageNo 属性
                        map.remove("pageSize");
                        map.remove("pageNo");
                    }

                    return JsonUtils.toPojo(map, clazz);
                } else {
                    HashMap<String, Object> map = new HashMap<>();
                    map.put("list", responseData);
                    return JsonUtils.toPojo(map, clazz);
                }

            } else {
                return buildErrorResponse((String) rspMap.get(Constants.Response.code), (String) rspMap.get(Constants.Response.message), (String) rspMap.get(Constants.Response.trace_id), clazz);
            }
        } catch (Exception ee) {
            log.error(ee.getMessage());
            return buildErrorResponse(ResponseCodeEnum.S_Common_NetworkError.getCode(), ResponseCodeEnum.S_Common_NetworkError.getMsg(), "", clazz);
        }
    }

    private <T> T buildErrorResponse(String errorCode, String errorMsg, String traceId, Class<T> clazz) {
        Map map = new LinkedHashMap();
        map.put("traceId", traceId);
        map.put("errorCode", errorCode);
        map.put("errorMsg", errorMsg);
        return JsonUtils.toPojo(map, clazz);
    }

    private Map<String, String> header(Map<String, String> headers, Endpoint endpoint, String requestBody) {
        Long timestamp = System.currentTimeMillis();

        headers.put("X-VERSION", "24041120");
        headers.put("User-Agent", "Actyve-Openapi-Java-SDK");
        headers.put("Request-Id", UUID.randomUUID().toString().replaceAll("-", ""));

        if (endpoint.getMethod().equalsIgnoreCase(Constants.HttpMethod.POST)) {
            String signContent = this.clientConfig.getAppId() + timestamp + requestBody;
            String sign = SignUtils.rsaSign(signContent, this.clientConfig.getPrivateKey());
            if (this.clientConfig.isDebug()) {
                log.info("Sign Actyve API:{},content:{},sign:{}", endpoint.getUrl(), signContent, sign);
            }
            headers.put("Sign", sign);
            headers.put("X-Timestamp", timestamp.toString());
        }

        if (endpoint.isAuthorization()) {
            headers.put("Authorization", "Bearer " + this.clientConfig.getAccessToken());
        }

        return headers;
    }

    private Class getResponseClass(ActyveRequest request) {
        ParameterizedType parameterizedType = (ParameterizedType) request.getClass().getGenericSuperclass();
        Type[] actualTypeArguments = parameterizedType.getActualTypeArguments();
        return (Class) actualTypeArguments[0];
    }
}
