package com.actyve.openapi.sdk.utils;


import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;


public class SignUtils {
    private static final Logger log = LoggerFactory.getLogger(SignUtils.class);
    /**
     * UTF-8字符集
     **/
    private static final String CHARSET_UTF8 = "UTF-8";

    private static final String SIGN_TYPE_RSA = "RSA";
    private static final String SIGN_ALGORITHMS = "SHA1WithRSA";

    /**
     * sha256WithRsa 算法请求类型
     */
    private static final String SIGN_TYPE_RSA2 = "RSA2";
    private static final String SIGN_SHA256RSA_ALGORITHMS = "SHA256WithRSA";

    /**
     * 获取签名值
     * @param content
     * @param privateKey
     * @return
     */
    public static String rsaSign(String content, String privateKey) {
        return rsaSign(content, privateKey, CHARSET_UTF8, SIGN_TYPE_RSA2);
    }

    /**
     * sha1WithRsa 加签
     *
     * @param content    签名内容
     * @param privateKey 私钥
     * @param charset    字符编码
     * @param signType   签名类型 RSA RSA2
     * @return String
     */
    public static String rsaSign(String content, String privateKey, String charset, String signType) {
        String algorithm;
        if (SIGN_TYPE_RSA.equals(signType)) {
            algorithm = SIGN_ALGORITHMS;
        } else if (SIGN_TYPE_RSA2.equals(signType)) {
            algorithm = SIGN_SHA256RSA_ALGORITHMS;
        } else {
            throw new RuntimeException("Sign Type is Not Support : signType=" + signType);
        }

        try {
            PrivateKey priKey = getPrivateKeyFromPKCS8(SIGN_TYPE_RSA, privateKey);

            java.security.Signature signature = java.security.Signature.getInstance(algorithm);
            signature.initSign(priKey);

            if (StringUtils.isEmpty(charset)) {
                signature.update(content.getBytes());
            } else {
                signature.update(content.getBytes(charset));
            }

            byte[] signed = signature.sign();

            return new String(Base64.encodeToByte(signed, false));
        } catch (InvalidKeySpecException ie) {
            throw new RuntimeException("RSA私钥格式不正确，请检查是否正确配置了PKCS8格式的私钥", ie);
        } catch (Exception e) {
            throw new RuntimeException("RSAcontent = " + content + "; charset = " + charset, e);
        }
    }

    private static PrivateKey getPrivateKeyFromPKCS8(String algorithm, String privateKey) throws Exception {
        if (privateKey == null || StringUtils.isEmpty(algorithm)) {
            return null;
        }
        KeyFactory keyFactory = KeyFactory.getInstance(algorithm);
        byte[] encodedKey = privateKey.getBytes();
        encodedKey = Base64.decode(encodedKey);
        return keyFactory.generatePrivate(new PKCS8EncodedKeySpec(encodedKey));
    }

    // ------------------------------------------- 验签 -------------------------------------------

    /**
     * 验签
     * @param content
     * @param sign
     * @param publicKey
     * @return
     */
    public static boolean rsaCheck(String content,String sign, String publicKey) {
        try {
            return rsaCheckContent(content,sign,publicKey, CHARSET_UTF8, SIGN_TYPE_RSA2);
        } catch (Exception e) {
            log.error("签名失败",e);
            return false;
        }
    }

    /**
     * 验签
     *
     * @param content   验签内容
     * @param sign      签名
     * @param publicKey 商户公钥
     * @param charset   字符编码
     * @param signType  签名类型
     * @return boolean
     */
    private static boolean rsaCheckContent(String content, String sign, String publicKey, String charset, String signType) {
        String algorithm;
        if (SIGN_TYPE_RSA.equals(signType)) {
            algorithm = SIGN_ALGORITHMS;
        } else if (SIGN_TYPE_RSA2.equals(signType)) {
            algorithm = SIGN_SHA256RSA_ALGORITHMS;
        } else {
            throw new RuntimeException("Sign Type is Not Support : signType=" + signType);
        }
        try {
            PublicKey pubKey = getPublicKeyFromX509(SIGN_TYPE_RSA, publicKey);
            java.security.Signature signature = java.security.Signature.getInstance(algorithm);
            signature.initVerify(pubKey);
            if (StringUtils.isEmpty(charset)) {
                signature.update(content.getBytes());
            } else {
                signature.update(content.getBytes(charset));
            }
            return signature.verify(Base64.decode(sign.getBytes()));
        } catch (Throwable e) {
            throw new RuntimeException("RSAcontent = " + content + ",sign=" + sign + ",charset = " + charset, e);
        }finally {
            System.out.println();
        }
    }

    private static PublicKey getPublicKeyFromX509(String algorithm, String publicKey) throws Exception {
        KeyFactory keyFactory = KeyFactory.getInstance(algorithm);
        byte[] encodedKey = publicKey.getBytes();
        encodedKey = Base64.decode(encodedKey);
        return keyFactory.generatePublic(new X509EncodedKeySpec(encodedKey));
    }
}
